<?php
/**
 * Plugin Name:       AI Gemini Keyword Reporter
 * Plugin URI:        https://example.com/
 * Description:       [v5.1] 帶密碼保護、整合性排程(指定時間啟動後週期執行)，可批次處理關鍵字並為每個結果獨立發信。
 * Version:           5.1.0
 * Author:            Your Name
 * Author URI:        https://example.com/
 * License:           GPL v2 or later
 * License URI:       https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain:       wp-gemini-keyword-reporter
 * Domain Path:       /languages
 */

if (!defined('ABSPATH')) exit;

class WPGeminiKeywordReporter {

    private $options;
    const LOG_OPTION_NAME = 'wgkr_logs';
    const REPORT_TRANSIENT_PREFIX = 'wgkr_report_';
    const REQUIRED_PASSWORD = '0936033684';
    const ONE_TIME_CRON_HOOK = 'wgkr_one_time_cron_hook';
    const RECURRING_CRON_HOOK = 'wgkr_recurring_cron_hook';

    public function __construct() {
        $this->options = get_option('wgkr_settings');
        add_action('admin_menu', [$this, 'add_admin_menu']);
        add_action('admin_init', [$this, 'settings_init']);
        add_action('init', [$this, 'add_rewrite_rules']);
        add_filter('query_vars', [$this, 'add_query_vars']);
        add_action('template_redirect', [$this, 'template_redirect_for_report']);
        add_action(self::RECURRING_CRON_HOOK, [$this, 'execute_recurring_task']);
        add_action(self::ONE_TIME_CRON_HOOK, [$this, 'execute_one_time_task']);
        add_filter('cron_schedules', [$this, 'add_custom_cron_schedule']);
        add_action('phpmailer_init', [$this, 'configure_smtp']);
        register_activation_hook(__FILE__, [$this, 'activate']);
        register_deactivation_hook(__FILE__, [$this, 'deactivate']);
    }

    public function activate() {
        $this->log_message('外掛已啟用。');
        $this->add_rewrite_rules();
        flush_rewrite_rules();
        $this->schedule_recurring_cron();
    }

    public function deactivate() {
        flush_rewrite_rules();
        wp_clear_scheduled_hook(self::RECURRING_CRON_HOOK);
        wp_clear_scheduled_hook(self::ONE_TIME_CRON_HOOK);
        $this->log_message('外掛已停用，所有排程已清除。');
    }
    
    private function is_password_correct() {
        $entered_password = $this->options['plugin_password'] ?? '';
        return $entered_password === self::REQUIRED_PASSWORD;
    }
    
    public function add_rewrite_rules() { add_rewrite_rule('^ai-report/([^/]+)/?$', 'index.php?wgkr_report_id=$matches[1]', 'top'); }
    public function add_query_vars($vars) { $vars[] = 'wgkr_report_id'; return $vars; }
    public function template_redirect_for_report() { if (get_query_var('wgkr_report_id')) { $this->render_report_page(); exit; } }

    public function add_admin_menu() {
        add_options_page('AI Gemini Reporter', 'AI Gemini Reporter', 'manage_options', 'wp_gemini_keyword_reporter', [$this, 'options_page_html']);
    }

    public function settings_init() {
        register_setting('wgkr_settings_group', 'wgkr_settings', [$this, 'sanitize_settings']);

        add_settings_section('wgkr_password_section', '功能密碼', fn() => print '<p>請輸入指定密碼以啟用外掛的核心功能。</p>', 'wp_gemini_keyword_reporter');
        add_settings_field('plugin_password', '密碼', [$this, 'render_field'], 'wp_gemini_keyword_reporter', 'wgkr_password_section', ['type' => 'password', 'id' => 'plugin_password', 'desc' => '']);
        
        if ($this->is_password_correct()) {
            add_settings_section('wgkr_main_section', '主要設定', null, 'wp_gemini_keyword_reporter');
            $api_key_desc = '請輸入您的 Google AI Gemini API 金鑰。 <a href="https://aistudio.google.com/app/apikey" target="_blank">點此前往 Google AI Studio 獲取金鑰</a>。<br>流程：點擊連結 -> 登入 Google 帳號 -> 點擊 "Create API key in new project"。';
            add_settings_field('api_key', 'Gemini API 金鑰', [$this, 'render_field'], 'wp_gemini_keyword_reporter', 'wgkr_main_section', ['type' => 'password', 'id' => 'api_key', 'desc' => $api_key_desc]);
            add_settings_field('gemini_model', 'Gemini 模型', [$this, 'render_field'], 'wp_gemini_keyword_reporter', 'wgkr_main_section', ['type' => 'text', 'id' => 'gemini_model', 'desc' => '預設使用 gemini-1.5-flash-latest 模型。']);
            add_settings_field('keywords', '關鍵字列表 (變數A)', [$this, 'render_field'], 'wp_gemini_keyword_reporter', 'wgkr_main_section', ['type' => 'textarea', 'id' => 'keywords', 'desc' => '每行一個關鍵字。']);
            add_settings_field('image_params', '圖片參數列表 (變數B)', [$this, 'render_field'], 'wp_gemini_keyword_reporter', 'wgkr_main_section', ['type' => 'textarea', 'id' => 'image_params', 'desc' => '每行一個參數，與上方關鍵字一一對應。若此列表較短，將會循環使用。']);
            add_settings_field('batch_size', '每次處理數量', [$this, 'render_field'], 'wp_gemini_keyword_reporter', 'wgkr_main_section', ['type' => 'number', 'id' => 'batch_size', 'desc' => '設定每次執行要處理的資料行數。將為每行資料獨立發送一封郵件。預設為 1。']);
            add_settings_field('ai_prompt', 'AI 指令 (Prompt)', [$this, 'render_field'], 'wp_gemini_keyword_reporter', 'wgkr_main_section', ['type' => 'textarea', 'id' => 'ai_prompt', 'desc' => '輸入您的 AI 指令。使用 <code>%%KEYWORD%%</code> (變數A) 和 <code>%%IMAGE_PARAM%%</code> (變數B) 作為佔位符。']);
            
            add_settings_section('wgkr_email_section', '郵件與跳轉設定', null, 'wp_gemini_keyword_reporter');
            add_settings_field('recipient_email', '收件人信箱', [$this, 'render_field'], 'wp_gemini_keyword_reporter', 'wgkr_email_section', ['type' => 'email', 'id' => 'recipient_email', 'desc' => '報告將會寄送到此信箱。']);
            add_settings_field('email_subject', '郵件主旨', [$this, 'render_field'], 'wp_gemini_keyword_reporter', 'wgkr_email_section', ['type' => 'text', 'id' => 'email_subject', 'desc' => '您可以在主旨中使用 <code>%%KEYWORD%%</code> 佔位符。']);
            add_settings_field('redirect_url', '複製後跳轉網址', [$this, 'render_field'], 'wp_gemini_keyword_reporter', 'wgkr_email_section', ['type' => 'url', 'id' => 'redirect_url', 'desc' => '在報告頁面點擊按鈕後，將會跳轉到此網址。']);

            add_settings_section('wgkr_recurring_cron_section', '週期性排程設定', null, 'wp_gemini_keyword_reporter');
            add_settings_field('cron_interval', '發送頻率 (小時)', [$this, 'render_field'], 'wp_gemini_keyword_reporter', 'wgkr_recurring_cron_section', ['type' => 'number', 'id' => 'cron_interval', 'desc' => '每隔幾小時發送一次報告。設為 0 可停用。']);
            
            add_settings_section('wgkr_smtp_section', 'SMTP 設定 (可選)', fn() => print '<p>如果您的主機無法正常發信，請填寫此區塊以使用外部 SMTP 伺服器發信。</p>', 'wp_gemini_keyword_reporter');
            add_settings_field('smtp_host', 'SMTP 主機', [$this, 'render_field'], 'wp_gemini_keyword_reporter', 'wgkr_smtp_section', ['type' => 'text', 'id' => 'smtp_host']);
            add_settings_field('smtp_port', 'SMTP 連接埠', [$this, 'render_field'], 'wp_gemini_keyword_reporter', 'wgkr_smtp_section', ['type' => 'number', 'id' => 'smtp_port']);
            add_settings_field('smtp_encryption', '加密方式', [$this, 'render_field'], 'wp_gemini_keyword_reporter', 'wgkr_smtp_section', ['type' => 'select', 'id' => 'smtp_encryption', 'options' => ['' => '無', 'ssl' => 'SSL', 'tls' => 'TLS']]);
            add_settings_field('smtp_username', 'SMTP 使用者名稱', [$this, 'render_field'], 'wp_gemini_keyword_reporter', 'wgkr_smtp_section', ['type' => 'text', 'id' => 'smtp_username']);
            add_settings_field('smtp_password', 'SMTP 密碼', [$this, 'render_field'], 'wp_gemini_keyword_reporter', 'wgkr_smtp_section', ['type' => 'password', 'id' => 'smtp_password']);
        }
    }

    public function render_field($args) {
        $id = $args['id'];
        $type = $args['type'];
        $value = $this->options[$id] ?? '';
        if ($id === 'gemini_model' && empty($value)) $value = 'gemini-1.5-flash-latest';
        $desc = $args['desc'] ?? '';
        $desc_html = $desc ? "<p class='description'>$desc</p>" : '';
        switch ($type) {
            case 'textarea': echo "<textarea id='$id' name='wgkr_settings[$id]' rows='8' class='large-text'>" . esc_textarea($value) . "</textarea>$desc_html"; break;
            case 'select':
                echo "<select id='$id' name='wgkr_settings[$id]'>";
                foreach ($args['options'] as $key => $label) echo "<option value='" . esc_attr($key) . "'" . selected($value, $key, false) . ">" . esc_html($label) . "</option>";
                echo "</select>$desc_html"; break;
            case 'number': echo "<input type='number' min='0' id='" . esc_attr($id) . "' name='wgkr_settings[" . esc_attr($id) . "]' value='" . esc_attr($value) . "' class='regular-text' placeholder='1'>$desc_html"; break;
            default: echo "<input type='" . esc_attr($type) . "' id='" . esc_attr($id) . "' name='wgkr_settings[" . esc_attr($id) . "]' value='" . esc_attr($value) . "' class='regular-text'>$desc_html"; break;
        }
    }
    
    public function sanitize_settings($input) {
        $new_input = $this->options ?: [];
        $map = [
            'keywords' => 'sanitize_textarea_field', 'image_params' => 'sanitize_textarea_field', 'ai_prompt' => 'sanitize_textarea_field',
            'recipient_email' => 'sanitize_email', 'redirect_url' => 'esc_url_raw',
            'cron_interval' => 'absint', 'smtp_port' => 'absint', 'batch_size' => 'absint',
        ];
        foreach ($input as $key => $value) {
            $new_input[$key] = isset($map[$key]) ? call_user_func($map[$key], $value) : sanitize_text_field(trim($value));
        }
        $this->schedule_recurring_cron(true, $new_input);
        $this->log_message('設定已儲存。');
        return $new_input;
    }

    public function options_page_html() {
        if (!current_user_can('manage_options')) return;

        if ($_SERVER['REQUEST_METHOD'] === 'POST' && !isset($_POST['submit'])) {
            if (isset($_POST['send_now'])) $this->execute_cron_task('now');
            elseif (isset($_POST['send_test_email'])) $this->execute_cron_task('test');
            elseif (isset($_POST['set_one_time_schedule'])) $this->handle_one_time_schedule_action();
            elseif (isset($_POST['cancel_one_time_schedule'])) $this->handle_one_time_schedule_action(true);
            elseif (isset($_POST['clear_log'])) {
                check_admin_referer('wgkr_clear_log_nonce');
                if($this->is_password_correct()) { delete_option(self::LOG_OPTION_NAME); $this->show_admin_notice('日誌已清除。', 'success'); }
            }
            elseif (isset($_POST['reset_index'])) {
                check_admin_referer('wgkr_reset_index_nonce');
                if ($this->is_password_correct()) {
                    $options = get_option('wgkr_settings');
                    $options['last_sent_keyword_index'] = -1;
                    update_option('wgkr_settings', $options);
                    $this->log_message('處理順序已由使用者手動重設。');
                    $this->show_admin_notice('處理順序已重設，下次將從第一個開始。', 'success');
                }
            }
        }
        $this->options = get_option('wgkr_settings');
        
        ?>
        <div class="wrap">
            <h1><?php echo esc_html(get_admin_page_title()); ?></h1>
            <form action="options.php" method="post"><?php settings_fields('wgkr_settings_group'); do_settings_sections('wp_gemini_keyword_reporter'); submit_button('儲存設定'); ?></form>
            
            <?php if (!$this->is_password_correct()): ?>
                <div class="notice notice-error" style="margin-top: 20px;"><p><strong>功能已鎖定：</strong>請在上方輸入正確的密碼並儲存，以解鎖所有功能。</p></div>
            <?php else: 
                $this->display_unlocked_content();
            endif; ?>
        </div>
        <?php
    }
    
    private function display_unlocked_content() {
        ?>
        <hr/>
        <h2>單次排程設定</h2>
        <p>設定一個未來的特定時間點，執行一次發送任務。執行後，系統將會根據下方的「週期性排程」設定，自動安排下一次的執行時間。</p>
        <form method="post">
            <?php wp_nonce_field('wgkr_one_time_schedule_nonce'); ?>
            <table class="form-table">
                <tbody>
                    <tr>
                        <th scope="row"><label for="one-time-schedule-time">指定執行時間</label></th>
                        <td>
                            <input type="datetime-local" id="one-time-schedule-time" name="one_time_schedule_time" value="<?php echo date('Y-m-d\TH:i', time() + 3600); ?>">
                            <p class="description"><strong>主機目前時間：</strong> <code><?php echo current_time('Y-m-d H:i:s'); ?></code> (時區: <?php echo wp_timezone_string(); ?>)</p>
                        </td>
                    </tr>
                </tbody>
            </table>
            <?php submit_button('設定單次排程', 'secondary', 'set_one_time_schedule', false); ?>
            <?php 
            $next_one_time = wp_next_scheduled(self::ONE_TIME_CRON_HOOK);
            if ($next_one_time) {
                echo ' <input type="submit" name="cancel_one_time_schedule" class="button" value="取消已設定的單次排程">';
                echo '<p><strong>已排程時間：</strong> <code>' . get_date_from_gmt(date('Y-m-d H:i:s', $next_one_time), 'Y-m-d H:i:s') . '</code></p>';
            }
            ?>
        </form>

        <hr/>
        <h2>手動操作</h2>
        <table class="form-table">
            <tbody>
                <?php
                    $keywords = isset($this->options['keywords']) ? array_values(array_filter(array_map('trim', explode("\n", $this->options['keywords'])))) : [];
                    $image_params = isset($this->options['image_params']) ? array_values(array_filter(array_map('trim', explode("\n", $this->options['image_params'])))) : [];
                    $last_index = $this->options['last_sent_keyword_index'] ?? -1;
                    $next_index = !empty($keywords) ? ((int)$last_index + 1) % count($keywords) : 0;
                    $next_keyword = !empty($keywords) ? $keywords[$next_index] : 'N/A';
                    $next_image_param = !empty($image_params) ? $image_params[$next_index % count($image_params)] : 'N/A';
                    $batch_size = !empty($this->options['batch_size']) ? absint($this->options['batch_size']) : 1;
                    $send_now_desc = "處理下一批「<b>" . esc_html($next_keyword) . "</b>」" . ($batch_size > 1 ? "等 {$batch_size} 行資料" : "資料") . "，為每個結果獨立發信，並推進序列。此操作會被記錄。";
                ?>
                <tr>
                    <th scope="row">下次將處理的資料行</th>
                    <td>
                        <p><strong>關鍵字 (A):</strong> <code><?php echo esc_html($next_keyword); ?></code></p>
                        <p><strong>圖片參數 (B):</strong> <code><?php echo esc_html($next_image_param); ?></code></p>
                        <form method="post" style="display: inline-block; margin-top: 10px;">
                            <?php wp_nonce_field('wgkr_reset_index_nonce'); ?>
                            <?php submit_button('重設順序', 'delete', 'reset_index', false); ?>
                        </form>
                        <p class="description">點擊「重設順序」會讓下次發送從列表中的第一行資料開始。</p>
                    </td>
                </tr>
                <tr><th scope="row">立即發送下一批報告</th><td><form method="post" style="display:inline-block;"><?php submit_button('立即發送下一批報告', 'primary', 'send_now', false); ?></form><p class="description"><?php echo $send_now_desc; ?></p></td></tr>
                <tr><th scope="row">發送測試郵件</th><td><form method="post" style="display:inline-block;"><?php submit_button('發送測試郵件', 'secondary', 'send_test_email', false); ?></form><p class="description">僅使用第一行資料發送一封測試郵件，不推進序列，不記錄。</p></td></tr>
            </tbody>
        </table>
        <hr/>
        <h2>執行日誌</h2>
        <div id="wgkr-log-viewer" style="height:300px;overflow-y:scroll;background:#f7f7f7;border:1px solid #ccc;padding:10px;font-family:monospace;white-space:pre-wrap;"><?php
            $logs = get_option(self::LOG_OPTION_NAME, []);
            echo empty($logs) ? '尚無日誌記錄。' : implode("\n", array_map('esc_html', array_reverse($logs)));
        ?></div>
        <form method="post" style="margin-top:10px;"><?php wp_nonce_field('wgkr_clear_log_nonce'); ?><?php submit_button('清除日誌', 'delete', 'clear_log', false); ?></form>
        <hr/>
        <h2>版本資訊與教學</h2>
        <div id="wgkr-info" style="background: #fff; border: 1px solid #c3c4c7; padding: 10px 20px;">
            <p><strong>版本號:</strong> 5.1.0</p>
            <h3>功能列表：</h3>
            <ul>
                <li>✅ <strong>雙變數系統:</strong> AI 指令可使用 %%KEYWORD%% 和 %%IMAGE_PARAM%% 兩個動態變數。</li>
                <li>✅ <strong>批次處理:</strong> 可自訂每次要處理的資料行數。</li>
                <li>✅ <strong>獨立郵件發送:</strong> 批次處理時，為每個成功的結果發送一封獨立的郵件。</li>
                <li>✅ <strong>整合性排程:</strong> 可設定「單次排程」在指定時間啟動，執行後自動銜接「週期性排程」。</li>
                <li>✅ <strong>順序重設:</strong> 可隨時將處理順序重置回第一筆資料。</li>
                <li>✅ <strong>一鍵複製與跳轉:</strong> 郵件中的連結導向專屬報告頁，可一鍵複製內容並跳轉。</li>
                <li>✅ <strong>SMTP支援:</strong> 可設定外部 SMTP 伺服器以確保郵件發送成功。</li>
                <li>✅ <strong>操作日誌:</strong> 記錄所有重要操作，方便追蹤與除錯。</li>
            </ul>
            <h3>簡易教學：</h3>
            <ol>
                <li><strong>填寫主要設定:</strong> 輸入您的 Gemini API 金鑰、關鍵字列表 (A)、圖片參數列表 (B) 和 AI 指令。</li>
                <li><strong>設定郵件:</strong> 填寫您的收件信箱、郵件主旨和複製後要跳轉的網址。</li>
                <li><strong>設定排程:</strong> 根據需求設定「週期性排程」或「單次排程」。單次排程會在執行後，自動接續週期性排程。</li>
                <li><strong>手動執行 (可選):</strong> 使用「手動操作」區塊的按鈕立即執行或測試任務。</li>
                <li><strong>檢查日誌:</strong> 在「執行日誌」區塊查看所有操作的詳細記錄。</li>
            </ol>
        </div>
        <?php
    }

    public function add_custom_cron_schedule($schedules) {
        $interval = $this->options['cron_interval'] ?? 0;
        if (absint($interval) > 0) $schedules['wgkr_custom_interval'] = ['interval' => absint($interval) * HOUR_IN_SECONDS, 'display'  => sprintf('每 %d 小時', $interval)];
        return $schedules;
    }
    
    private function schedule_recurring_cron($reschedule = false, $settings = null) {
        if ($reschedule) {
            wp_clear_scheduled_hook(self::RECURRING_CRON_HOOK);
        }
        
        $options = $settings ?: $this->options;
        // Do not schedule a recurring event if a one-time event is already scheduled.
        if (wp_next_scheduled(self::ONE_TIME_CRON_HOOK)) {
             $this->log_message('偵測到已設定的單次排程，週期性排程將在單次排程執行後啟動。');
             return;
        }

        if (wp_next_scheduled(self::RECURRING_CRON_HOOK)) return;

        if ($this->is_password_correct() && isset($options['cron_interval']) && absint($options['cron_interval']) > 0) {
            wp_schedule_event(time(), 'wgkr_custom_interval', self::RECURRING_CRON_HOOK);
            $this->log_message(sprintf('週期性排程已設定。下次執行: %s', wp_date('Y-m-d H:i:s', wp_next_scheduled(self::RECURRING_CRON_HOOK))));
        } else {
            $this->log_message('週期性排程已停用 (密碼不正確或頻率為0)。');
        }
    }

    private function handle_one_time_schedule_action($cancel = false) {
        check_admin_referer('wgkr_one_time_schedule_nonce');

        // Always clear both schedules first when this action is taken
        wp_clear_scheduled_hook(self::ONE_TIME_CRON_HOOK);
        wp_clear_scheduled_hook(self::RECURRING_CRON_HOOK);
        
        if ($cancel) {
            $this->log_message("單次排程已被手動取消。");
            $this->show_admin_notice("已取消設定的單次排程。", "info");
            // After canceling, try to set up the recurring schedule again
            $this->schedule_recurring_cron();
            return;
        }

        $schedule_time_local = sanitize_text_field($_POST['one_time_schedule_time']);
        if (empty($schedule_time_local)) {
            $this->show_admin_notice("錯誤：請指定一個有效的時間。", "error");
            return;
        }

        $schedule_timestamp_utc = strtotime(get_gmt_from_date($schedule_time_local));
        if (!$schedule_timestamp_utc || $schedule_timestamp_utc < time()) {
             $this->show_admin_notice("錯誤：排程時間必須是未來的時間。", "error");
            return;
        }

        wp_schedule_single_event($schedule_timestamp_utc, self::ONE_TIME_CRON_HOOK);
        $this->log_message("已設定單次排程於 " . $schedule_time_local . " (網站時區)。週期性排程將在此之後啟動。");
        $this->show_admin_notice("已成功設定單次排程！", "success");
    }

    public function execute_recurring_task() { $this->execute_cron_task('recurring_cron'); }
    
    public function execute_one_time_task() { 
        $this->execute_cron_task('one_time_cron');
        // After one-time task execution, immediately start the recurring schedule
        $this->schedule_recurring_cron(true); // `true` to force reschedule
    }

    public function execute_cron_task($mode = 'cron') {
        if (!$this->is_password_correct()) {
            $this->log_message("[$mode] 任務中止：密碼不正確。");
            return;
        }

        $log_prefix = "[".strtoupper($mode)."]";
        $is_test = ($mode === 'test');
        if (!$is_test) $this->log_message("$log_prefix 任務開始執行。");

        $options = get_option('wgkr_settings');
        if (empty($options['api_key']) || empty($options['keywords']) || empty($options['ai_prompt']) || empty($options['recipient_email'])) {
            $error_msg = "$log_prefix 任務中止：API金鑰、關鍵字列表(A)、AI指令和收件人信箱為必填欄位。";
            if (!$is_test) $this->log_message($error_msg);
            if ($is_test || $mode === 'now') $this->show_admin_notice($error_msg, 'error');
            return;
        }

        $keywords = array_values(array_filter(array_map('trim', explode("\n", $options['keywords']))));
        $image_params = isset($options['image_params']) ? array_values(array_filter(array_map('trim', explode("\n", $options['image_params'])))) : [];
        $batch_size = ($is_test || empty($options['batch_size'])) ? 1 : absint($options['batch_size']);
        $last_index = $options['last_sent_keyword_index'] ?? -1;
        $model = !empty($options['gemini_model']) ? $options['gemini_model'] : 'gemini-1.5-flash-latest';
        
        $final_keyword_index = $last_index;
        $sent_count = 0;

        for ($i = 0; $i < $batch_size; $i++) {
            $current_index = ((int)$last_index + 1 + $i);
            if ($current_index >= count($keywords)) {
                 if ($is_test || $i > 0) {
                    if (!$is_test && $i > 0) $this->log_message("$log_prefix 已處理完所有關鍵字，本次批次提前結束。");
                    break;
                 }
                 $current_index %= count($keywords); 
            }
            
            $keyword = $keywords[$current_index];
            $image_param = !empty($image_params) ? $image_params[$current_index % count($image_params)] : '';

            if (!$is_test) $this->log_message("$log_prefix 準備處理批次中的資料 #${current_index}: A='$keyword', B='$image_param'");

            $replacements = ['%%KEYWORD%%' => $keyword, '%%IMAGE_PARAM%%' => $image_param];
            $final_prompt = str_replace(array_keys($replacements), array_values($replacements), $options['ai_prompt']);
            $result = $this->call_gemini_api($final_prompt, $options['api_key'], $model);

            if (strpos($result, "API ") === 0) {
                if (!$is_test) $this->log_message("$log_prefix $result for A='$keyword' - 已跳過此行。");
                continue; 
            }
            
            $email_subject = str_replace('%%KEYWORD%%', $keyword, $options['email_subject']);
            $report_id = wp_generate_password(24, false);
            $report_url = home_url('/ai-report/' . $report_id . '/');
            set_transient(self::REPORT_TRANSIENT_PREFIX . $report_id, ['content' => $result, 'keyword' => $keyword, 'image_param' => $image_param], DAY_IN_SECONDS);
            
            $email_body = "<h1>AI 關鍵字報告</h1><p>關於「<strong>".esc_html($keyword)."</strong>」的報告已生成。</p><p>請點擊下方按鈕前往專屬頁面查看內容，並使用一鍵複製功能：</p>";
            $email_body .= "<p style='text-align:center; padding: 20px 0;'><a href='".esc_url($report_url)."' style='display:inline-block;padding:12px 24px;font-size:16px;font-weight:bold;color:#ffffff;background-color:#0073aa;text-decoration:none;border-radius:5px;'>查看報告與複製內容</a></p>";
            $email_body .= "<p><small>如果按鈕無效，請直接複製以下連結到瀏覽器開啟：<br>".esc_url($report_url)."</small></p>";
        
            $sent = wp_mail($options['recipient_email'], $email_subject, $email_body, ['Content-Type: text/html; charset=UTF-8']);

            if ($sent) {
                $final_keyword_index = $current_index;
                $sent_count++;
                if (!$is_test) $this->log_message("$log_prefix 已成功為 '$keyword' 發送郵件。");
            } else {
                 if (!$is_test) $this->log_message("$log_prefix 為 '$keyword' 發送郵件失敗。");
            }
        }
        
        if ($sent_count > 0) {
            if (!$is_test) {
                $options['last_sent_keyword_index'] = $final_keyword_index;
                update_option('wgkr_settings', $options);
                $this->log_message("$log_prefix 批次任務完成，共成功發送 $sent_count 封郵件。");
            }
            if ($is_test || $mode === 'now') $this->show_admin_notice("任務完成，成功發送 $sent_count 封郵件。請檢查您的收件匣。", 'success');
        } else {
            if (!$is_test) $this->log_message("$log_prefix 批次任務完成，但未成功發送任何郵件。");
            if ($is_test || $mode === 'now') $this->show_admin_notice('任務完成，但沒有成功發送任何郵件，請檢查日誌。', 'warning');
        }
    }

    private function call_gemini_api($prompt, $api_key, $model) {
        $url = "https://generativelanguage.googleapis.com/v1beta/models/{$model}:generateContent?key=" . $api_key;
        $response = wp_remote_post($url, ['body' => json_encode(['contents' => [['parts' => [['text' => $prompt]]]]]), 'headers' => ['Content-Type' => 'application/json'], 'timeout' => 90]);
        if (is_wp_error($response)) return "API 請求失敗: " . $response->get_error_message();
        $data = json_decode(wp_remote_retrieve_body($response));
        return $data->candidates[0]->content->parts[0]->text ?? ($data->error->message ? "API 返回錯誤: ".$data->error->message : "API 返回未知錯誤。");
    }

    public function configure_smtp($phpmailer) {
        if (empty($this->options['smtp_host'])) return;
        $phpmailer->isSMTP();
        $phpmailer->Host = $this->options['smtp_host'];
        $phpmailer->SMTPAuth = true;
        $phpmailer->Port = absint($this->options['smtp_port'] ?: 587);
        $phpmailer->Username = $this->options['smtp_username'] ?? '';
        $phpmailer->Password = $this->options['smtp_password'] ?? '';
        $phpmailer->SMTPSecure = $this->options['smtp_encryption'] ?? 'tls';
        $phpmailer->From = $phpmailer->Username;
        $phpmailer->FromName = get_bloginfo('name');
    }

    private function log_message($message) {
        $logs = get_option(self::LOG_OPTION_NAME, []);
        $logs[] = wp_date('[Y-m-d H:i:s] ') . $message;
        if (count($logs) > 200) $logs = array_slice($logs, -200);
        update_option(self::LOG_OPTION_NAME, $logs, false);
    }
    
    private function show_admin_notice($message, $type = 'info') {
        add_action('admin_notices', fn() => printf('<div class="notice notice-%s is-dismissible"><p>%s</p></div>', esc_attr($type), esc_html($message)));
    }
    
    public function render_report_page() {
        $report_id = get_query_var('wgkr_report_id');
        $report_data = get_transient(self::REPORT_TRANSIENT_PREFIX . $report_id);
        header('Content-Type: text/html; charset=utf-8');
        ?>
        <!DOCTYPE html><html <?php language_attributes(); ?>><head><meta charset="<?php bloginfo('charset'); ?>"><meta name="viewport" content="width=device-width, initial-scale=1"><title>AI 報告</title><?php wp_head(); ?><style>body{font-family:-apple-system,BlinkMacSystemFont,"Segoe UI",Roboto,Oxygen-Sans,Ubuntu,Cantarell,"Helvetica Neue",sans-serif;background-color:#f0f2f5;color:#1c1e21;line-height:1.6;margin:0;padding:20px}.container{max-width:800px;margin:40px auto;background-color:#fff;border:1px solid #dddfe2;border-radius:8px;box-shadow:0 2px 4px rgba(0,0,0,.1)}.header{background-color:#0073aa;color:#fff;padding:20px;border-top-left-radius:8px;border-top-right-radius:8px}.header h1{margin:0;font-size:24px}.content{padding:20px 30px 30px}.content-box{border:1px solid #e0e0e0;padding:20px;margin-top:20px;border-radius:6px;background-color:#f9f9f9}#ai-content{white-space:pre-wrap;font-size:16px}.actions{margin-top:30px;text-align:center}.button{display:inline-block;padding:12px 24px;font-size:16px;font-weight:bold;color:#fff!important;background-color:#28a745;text-decoration:none;border-radius:5px;cursor:pointer;border:none;transition:background-color .3s}.button:hover{background-color:#218838}.notice{padding:10px;background-color:#d4edda;color:#155724;border-radius:4px;text-align:center;margin-top:15px;display:none}</style></head>
        <body><div class="container"><div class="header"><h1>AI 生成內容報告</h1></div><div class="content">
        <?php if ($report_data): ?>
            <h2>關鍵字 (A)：<?php echo esc_html($report_data['keyword']); ?></h2>
            <p style="color:#555;"><strong>使用的圖片參數 (B)：</strong><?php echo esc_html($report_data['image_param']); ?></p>
            <div class="content-box"><div id="ai-content"><?php echo wpautop(esc_html($report_data['content'])); ?></div></div>
            <div class="actions"><button id="copy-btn" class="button">一鍵複製內容並前往指定網址</button><div id="copy-notice" class="notice">內容已複製到剪貼簿！</div></div>
            <script>
                document.getElementById('copy-btn').addEventListener('click', function(){
                    const textToCopy = <?php echo json_encode($report_data['content']); ?>;
                    const redirectUrl = <?php echo json_encode(get_option('wgkr_settings')['redirect_url'] ?? ''); ?>;
                    navigator.clipboard.writeText(textToCopy).then(() => {
                        document.getElementById('copy-notice').style.display = 'block';
                        setTimeout(() => { if(redirectUrl) window.open(redirectUrl, '_blank'); }, 500);
                    }).catch(err => { console.error('無法複製: ', err); alert('複製失敗，請手動複製。'); });
                });
            </script>
        <?php else: ?>
            <h2>報告不存在或已過期</h2><p>此報告的連結可能不正確，或是內容已經超過 24 小時的保留期限而被系統自動清除。</p>
        <?php endif; ?>
        </div></div><?php wp_footer(); ?></body></html>
        <?php
    }
}

new WPGeminiKeywordReporter();